import json
import re
import xml.dom.minidom
from xml.etree.ElementTree import Element, SubElement, tostring
import base64
import os

# === CONFIGURATION ===
INPUT_JSON = "deck.json"           # ton fichier CrowdAnki
MEDIA_DIR = "media"                # dossier contenant les images (à côté du JSON)
OUTPUT_XML = "deck_to_moodle.xml"  # fichier Moodle généré
CATEGORY = "$course$/Nouveau"

# === CHARGEMENT DU DECK JSON ===
with open(INPUT_JSON, "r", encoding="utf-8") as f:
    data = json.load(f)

notes = data.get("notes", [])
print(f"Nombre de cartes détectées : {len(notes)}")

# === FONCTION D’ENCODAGE DES IMAGES EN BASE64 ===
def embed_images_in_html(html_text):
    def replace_img(match):
        img_file = match.group(1)
        img_path = os.path.join(MEDIA_DIR, img_file)
        if os.path.exists(img_path):
            with open(img_path, "rb") as img_f:
                encoded = base64.b64encode(img_f.read()).decode("utf-8")
                ext = os.path.splitext(img_file)[1][1:]  # ex: png, jpg
                return f'<img src="data:image/{ext};base64,{encoded}">'
        else:
            return f'<img src="{img_file}">'  # image manquante
    return re.sub(r'<img src="([^"]+)"', replace_img, html_text)

# === CONSTRUCTION DU QUIZ XML ===
quiz = Element("quiz")

# Catégorie Moodle
question_category = SubElement(quiz, "question", type="category")
category = SubElement(question_category, "category")
SubElement(category, "text").text = CATEGORY

# === TRAITEMENT DES CARTES ===
for i, note in enumerate(notes, start=1):
    fields = note.get("fields", {})
    if not fields:
        continue

    # ✅ Correction ici : compatibilité liste ou dictionnaire
    field_values = fields if isinstance(fields, list) else list(fields.values())

    question = field_values[0].strip() if len(field_values) > 0 else ""
    answer = field_values[1].strip() if len(field_values) > 1 else ""

    if not question or not answer:
        continue

    # Nettoyage léger (espaces et indices chimiques)
    question = re.sub(r"\s+", " ", question)
    answer = re.sub(r"\s+", " ", answer)
    sub_map = {"₂": "2", "₃": "3", "₄": "4", "₅": "5", "₆": "6", "₇": "7",
               "₈": "8", "₉": "9", "₀": "0", "₁": "1", "→": "->"}
    for k, v in sub_map.items():
        question = question.replace(k, v)
        answer = answer.replace(k, v)

    # Intégration des images en Base64
    question = embed_images_in_html(question)
    answer = embed_images_in_html(answer)

    # Création de la question Moodle
    q = SubElement(quiz, "question", type="multichoice")
    name = SubElement(q, "name")
    SubElement(name, "text").text = f"Carte {i}"

    questiontext = SubElement(q, "questiontext", format="html")
    SubElement(questiontext, "text").text = (
        f"<p><b>{question}</b></p>"
        f"<p><i>Réfléchis, puis clique sur <b>Vérifier</b> pour voir la réponse.</i></p>"
    )

    generalfeedback = SubElement(q, "generalfeedback", format="html")
    SubElement(generalfeedback, "text").text = answer

    SubElement(q, "defaultgrade").text = "1.0"
    SubElement(q, "penalty").text = "0.0"
    SubElement(q, "hidden").text = "0"
    SubElement(q, "single").text = "true"
    SubElement(q, "shuffleanswers").text = "false"
    SubElement(q, "answernumbering").text = "none"

    # Réponses
    ans1 = SubElement(q, "answer", fraction="100", format="html")
    SubElement(ans1, "text").text = "Voir la réponse"
    fb1 = SubElement(ans1, "feedback", format="html")
    SubElement(fb1, "text").text = answer

    ans2 = SubElement(q, "answer", fraction="0", format="html")
    SubElement(ans2, "text").text = "Je ne sais pas"
    fb2 = SubElement(ans2, "feedback", format="html")
    SubElement(fb2, "text").text = "Réfléchis encore un peu, puis clique sur <b>Voir la réponse</b>."

# === SAUVEGARDE DU XML ===
xml_str = xml.dom.minidom.parseString(tostring(quiz)).toprettyxml(indent="  ")

with open(OUTPUT_XML, "w", encoding="utf-8") as f:
    f.write(xml_str)

print(f"\n✅ Fichier Moodle XML généré avec images intégrées : {OUTPUT_XML}")
